//
//  ItemGroups.swift
//  Do It
//
//  Created by Jim Dovey on 2/11/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI
import CoreData

enum TodoItemGroup {
    case today
    case scheduled
    case overdue
    case all
    
    var title: LocalizedStringKey {
        switch self {
        case .today: return "Today"
        case .scheduled: return "Scheduled"
        case .overdue: return "Overdue"
        case .all: return "All"
        }
    }
    
    var color: Color {
        switch self {
        case .today: return .blue
        case .scheduled: return .orange
        case .overdue: return .red
        case .all: return Color(UIColor.darkGray)
        }
    }
    
    private var image: some View {
        switch self {
        case .today:
            return Image(systemName: "calendar")
                .offset(x: 0, y: 0)
        case .scheduled:
            return Image(systemName: "clock.fill")
                .offset(x: 0, y: 0)
        case .overdue:
            return Image(systemName: "exclamationmark.triangle.fill")
                .offset(x: 0, y: -2)
        case .all:
            return Image(systemName: "tray.fill")
                .offset(x: 0, y: -1)
        }
    }
    
    var icon: some View {
        self.image
            .font(.system(.body, design: .rounded))
            .foregroundColor(.white)
            .padding(10)
            .background(color.clipShape(Circle()))
    }
    
    var fetchPredicate: NSPredicate? {
        switch self {
        case .today:
            let todayComponents = Calendar.current
                .dateComponents([.day, .month, .year], from: Date())
            let today = Calendar.current.date(from: todayComponents)!
            let tomorrow = Date(timeInterval: 60 * 60 * 24, since: today)

//            return NSPredicate(format: "date >= %@ AND date <= %@ AND (completed > %@ OR completed == nil)",
//                               today as NSDate, tomorrow as NSDate, NSDate())

            let after = NSComparisonPredicate(
                leftExpression: NSExpression(forKeyPath: \TodoItem.date),
                rightExpression: NSExpression(forConstantValue: today),
                modifier: .direct,
                type: .greaterThanOrEqualTo,
                options: [])
            let before = NSComparisonPredicate(
                leftExpression: after.leftExpression,
                rightExpression: NSExpression(forConstantValue: tomorrow),
                modifier: .direct,
                type: .lessThan,
                options: [])

            return NSCompoundPredicate(
                andPredicateWithSubpredicates: [
                    after, before, TodoItem.notCompletePredicate])

        case .scheduled:
//            return NSPredicate(format: "date != nil")
            return NSComparisonPredicate(
                leftExpression: NSExpression(forKeyPath: \TodoItem.date),
                rightExpression: NSExpression(forConstantValue: nil),
                modifier: .direct,
                type: .notEqualTo,
                options: [])

        case .overdue:
//            return NSPredicate(format: "date < %@", Date() as NSDate)
            return NSComparisonPredicate(
                leftExpression: NSExpression(forKeyPath: \TodoItem.date),
                rightExpression: NSExpression(forConstantValue: Date()),
                modifier: .direct,
                type: .lessThan,
                options: [])

        case .all:
            return nil
        }
    }
    
    var fetchRequest: NSFetchRequest<TodoItem> {
        let request: NSFetchRequest<TodoItem> = TodoItem.fetchRequest()
        request.fetchBatchSize = 25
        request.predicate = self.fetchPredicate
        return request
    }
}
